<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class Currency extends Model
{
    protected $fillable = [
        'code', 'name', 'chart_code', 'profit_percentage', 'enabled',
        'rate', 'old_bid', 'old_ask', 'bid', 'ask', 'leverage'
    ];

    protected $casts = [
        'rate' => 'decimal:8',
        'old_bid' => 'decimal:8',
        'old_ask' => 'decimal:8',
        'bid' => 'decimal:8',
        'ask' => 'decimal:8',
        'leverage' => 'integer'
    ];

    protected $appends = ['has_bid_changed', 'has_ask_changed'];

    public function rates()
    {
        return $this->hasMany('App\Models\Rate');
    }

    public function type()
    {
        return $this->belongsTo('App\Models\CurrencyType');
    }

    public function iconExists()
    {
        $iconPath = 'assets/img/icon/asset-' . str_replace(['/', '#'], '', $this->code) . '.svg';
        return file_exists(public_path($iconPath));
    }

    public function getIconUrl()
    {
        return asset('assets/img/icon/asset-' . str_replace(['/', '#'], '', $this->code) . '.svg');
    }

    public function getFormattedRateAttribute()
    {
        return $this->rate ? number_format($this->rate, 2) : null;
    }

    public function getFormattedBidAttribute()
    {
        $decimals = 2;
        
        if ($this->type->code == 'currencies') {
            $decimals = 5;
        }

        if (str_contains($this->code, 'JPY') && $this->type->code == 'currencies') {
            $decimals = 3;
        }

        return $this->bid ? number_format($this->bid, $decimals) : null;
    }

    public function getFormattedAskAttribute()
    {
        $decimals = 2;
        
        if ($this->type->code == 'currencies') {
            $decimals = 5;
        }

        if (str_contains($this->code, 'JPY') && $this->type->code == 'currencies') {
            $decimals = 3;
        }

        return $this->ask ? number_format($this->ask, $decimals) : null;
    }

    public function getChangeClass()
    {
        if ($this->bid === null || $this->ask === null) {
            return '';
        }

        $bidChange = $this->old_bid < $this->bid;
        $askChange = $this->old_ask < $this->ask;

        if ($bidChange || $askChange) {
            return 'price-up';
        } elseif ($this->old_bid > $this->bid || $this->old_ask > $this->ask) {
            return 'price-down';
        }

        return '';
    }

    public function getChange()
    {
        if ($this->bid === null || $this->ask === null) {
            return 'n/a';
        }

        $bidChange = $this->old_bid < $this->bid;
        $askChange = $this->old_ask < $this->ask;

        if ($bidChange || $askChange) {
            return '+';
        } elseif ($this->old_bid > $this->bid || $this->old_ask > $this->ask) {
            return '-';
        }

        return '=';
    }

    public function getHasBidChangedAttribute()
    {
        return $this->old_bid !== null && $this->bid !== null && $this->old_bid != $this->bid;
    }

    public function getHasAskChangedAttribute()
    {
        return $this->old_ask !== null && $this->ask !== null && $this->old_ask != $this->ask;
    }
}
